<?php

namespace MbeGroup\Notification\Command;

use Illuminate\Console\Command;
use MbeGroup\Notification\Models\Notification;
use MbeGroup\Notification\Jobs\ProcessNotificationJob;
use App\ValueObject\Status;
use App\Models\CandidateProfile;

class CheckNotificationsCommand extends Command
{
    protected $signature = 'notifications:check';

    public function handle(): void
    {
        $userIds = Notification::where('status', Status::ACTIVE)
            ->raw(fn($collection) => $collection->distinct('user_id', ['status' => Status::ACTIVE]));

        CandidateProfile::whereIn('user_id', $userIds)
            ->each(function (CandidateProfile $candidate) {
                if (!$this->shouldSendForUser($candidate)) {
                    return;
                }
                ProcessNotificationJob::dispatch($candidate);
            });
    }

    private function shouldSendForUser(CandidateProfile $candidate): bool
    {
        if (blank($candidate->notification_frequency)) {
            return false;
        }

        $lastSentAt = Notification::where('user_id', $candidate->user_id)
            ->where('status', Status::ACTIVE)
            ->whereNotNull('last_sent_at')
            ->orderByDesc('last_sent_at')
            ->value('last_sent_at');

        if (is_null($lastSentAt)) {
            return true;
        }

        return \Carbon\Carbon::parse($lastSentAt)
            ->addDays($candidate->notification_frequency)
            ->isPast();
    }
}
