<?php

namespace MbeGroup\Notification\Jobs;

use MbeGroup\Notification\Models\Notification;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Support\Facades\Mail;
use MbeGroup\Notification\Factories\NotificationProviderFactory;
use Illuminate\Bus\Queueable;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use App\Models\CandidateProfile;
use App\ValueObject\Status;
use MbeGroup\Notification\Mail\CombinedNotificationMailable;


final class ProcessNotificationJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public function __construct(private CandidateProfile $candidate) {}

    public function handle(NotificationProviderFactory $factory): void
    {
        $notifications = Notification::where('user_id', $this->candidate->user_id)
            ->where('status', Status::ACTIVE)
            ->get();

        $email   = $this->candidate->user->email;
        $results = $notifications->map(function (Notification $notification) use ($factory) {
            $provider = $factory->make($notification->type);

            $items = $provider->getItems(
                query         : $notification->query,
                publishedAfter: $notification->last_sent_at,
            );

            return [
                'notification' => $notification,
                'items'        => $items,
            ];
        })->filter(fn($result) => $result['items']->isNotEmpty());

        $notifications
            ->filter(fn($n) => !$results->contains(fn($r) => $r['notification']->id === $n->id))
            ->each(fn($n) => $n->update(['last_queried_at' => now()]));

        if ($results->isEmpty()) {
            return;
        }

        Mail::to($email)
            ->queue(new CombinedNotificationMailable($this->candidate, $results->values()));

        $now = now();
        $results->each(fn($result) => $result['notification']->update([
            'last_sent_at'    => $now,
            'last_queried_at' => $now,
        ]));
    }
}
