<?php

namespace MbeGroup\Product\Filament\Resources;

use Filament\Resources\Resource;
use Filament\Schemas\Schema;
use Filament\Tables;
use Filament\Tables\Table;
use Filament\Tables\Filters\SelectFilter;
use MbeGroup\Product\Filament\Resources\ProductStatisticsResource\Pages\ListProductStatistics;
use MbeGroup\Product\Models\ProductStatistics;
use App\ValueObject\Status;
use MbeGroup\Employer\Models\Employer;

class ProductStatisticsResource extends Resource
{
    protected static ?string $model = ProductStatistics::class;
    public static function getNavigationGroup(): string
    {
        return __('product::filament.navigation.group.label');
    }

    public static function getNavigationLabel(): string
    {
        return __('product::filament.resources.product_statistics.navigation_label');
    }

    public static function getModelLabel(): string
    {
        return __('product::filament.resources.product_statistics.model_label');
    }

    public static function getPluralModelLabel(): string
    {
        return __('product::filament.resources.product_statistics.plural_model_label');
    }

    public static function form(Schema $schema): Schema
    {
        return $schema->schema([]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('employer_name')
                    ->label(__('product::filament.resources.product_statistics.table.columns.employer_name'))
                    ->getStateUsing(function ($record) {
                        $employer = Employer::find($record->employer_id);
                        return $employer ? $employer->name : 'Brak danych';
                    })
                    ->searchable(false)
                    ->sortable(false),
                Tables\Columns\TextColumn::make('product.name')
                    ->label(__('product::filament.resources.product_statistics.table.columns.product_name'))
                    ->searchable()
                    ->sortable(),
                Tables\Columns\TextColumn::make('quantity')
                    ->label(__('product::filament.resources.product_statistics.table.columns.quantity'))
                    ->sortable(),
                Tables\Columns\TextColumn::make('price')
                    ->label(__('product::filament.resources.product_statistics.table.columns.price'))
                    ->money('PLN')
                    ->sortable(),
                Tables\Columns\TextColumn::make('created_at')
                    ->label(__('product::filament.resources.product_statistics.table.columns.created_at'))
                    ->dateTime()
                    ->sortable(),
                Tables\Columns\TextColumn::make('valid_from')
                    ->label(__('product::filament.resources.product_statistics.table.columns.valid_from'))
                    ->date()
                    ->sortable(),
                Tables\Columns\TextColumn::make('valid_to')
                    ->label(__('product::filament.resources.product_statistics.table.columns.valid_to'))
                    ->date()
                    ->sortable(),
                Tables\Columns\IconColumn::make('status')
                    ->label(__('product::filament.resources.product_statistics.table.columns.status'))
                    ->state(fn ($record) => Status::int($record->status))
                    ->icon(fn ($state) => Status::icon($state))
                    ->color(fn ($state) => Status::color($state))
                    ->tooltip(fn ($state) => Status::label($state))
                    ->sortable(),
            ])
            ->filters([
                SelectFilter::make('status')
                    ->label(__('product::filament.resources.product_statistics.filters.status'))
                    ->options(Status::getLabels()),
                SelectFilter::make('employer_id')
                    ->label(__('product::filament.resources.product_statistics.filters.employer'))
                    ->options(function () {
                        return Employer::all()
                            ->pluck('name', '_id')
                            ->toArray();
                    })
                    ->searchable(),
                SelectFilter::make('product_id')
                    ->label(__('product::filament.resources.product_statistics.filters.product'))
                    ->relationship('product', 'name')
                    ->searchable()
                    ->preload(),
            ])
            ->defaultSort('created_at', 'desc')
            ->recordActions([
                // Tylko wyświetlanie, brak akcji edycji
            ])
            ->toolbarActions([
                // Brak akcji tworzenia nowych rekordów
            ])
            ->modifyQueryUsing(function ($query) {
                // Wyświetlaj wszystkie rekordy (aktywne i nieaktywne) - to jest historia
                // Ładuj tylko relację z produktem (MySQL), employer jest w MongoDB
                return $query->with(['product']);
            });
    }

    public static function canCreate(): bool
    {
        return false; // Wyłączenie możliwości tworzenia nowych rekordów
    }

    public static function getRelations(): array
    {
        return [];
    }

    public static function getPages(): array
    {
        return [
            'index' => ListProductStatistics::route('/'),
        ];
    }
}
