<?php

namespace Tests\Unit;

use MbeGroup\Product\Models\ProductStatistics;
use PHPUnit\Framework\TestCase;
use ReflectionClass;

class ProductStatisticsTest extends TestCase
{
    /**
     * Test tworzenia nowej instancji ProductStatistics
     */
    public function test_can_create_product_statistics_instance(): void
    {
        $productStatistics = new ProductStatistics();
        
        $this->assertInstanceOf(ProductStatistics::class, $productStatistics);
    }

    /**
     * Test sprawdzający czy klasa ma odpowiednie właściwości
     */
    public function test_product_statistics_class_properties(): void
    {
        $reflection = new ReflectionClass(ProductStatistics::class);
        
        // Sprawdzenie czy klasa dziedziczy po Illuminate\Database\Eloquent\Model
        $this->assertTrue($reflection->hasProperty('table'));
        $this->assertTrue($reflection->hasProperty('fillable'));
    }

    /**
     * Test sprawdzający wypełnialne pola
     */
    public function test_fillable_attributes(): void
    {
        $productStatistics = new ProductStatistics();
        $reflection = new ReflectionClass($productStatistics);
        
        $fillableProperty = $reflection->getProperty('fillable');
        $fillableProperty->setAccessible(true);
        $fillable = $fillableProperty->getValue($productStatistics);
        
        $expectedFillable = [
            'employer_id',
            'product_id',
            'quantity',
            'price',
            'valid_from',
            'valid_to',
            'status',
        ];
        $this->assertEquals($expectedFillable, $fillable);
    }

    /**
     * Test sprawdzający nazwę tabeli
     */
    public function test_table_name(): void
    {
        $productStatistics = new ProductStatistics();
        $reflection = new ReflectionClass($productStatistics);
        
        $tableProperty = $reflection->getProperty('table');
        $tableProperty->setAccessible(true);
        $table = $tableProperty->getValue($productStatistics);
        
        $this->assertEquals('product_statistics', $table);
    }

    /**
     * Test sprawdzający namespace klasy
     */
    public function test_class_namespace(): void
    {
        $productStatistics = new ProductStatistics();
        $reflection = new ReflectionClass($productStatistics);
        
        $this->assertEquals('MbeGroup\Product\Models', $reflection->getNamespaceName());
        $this->assertEquals('ProductStatistics', $reflection->getShortName());
    }

    /**
     * Test czy klasa implementuje odpowiednie metody (sprawdzenie czy dziedziczy po Model)
     */
    public function test_extends_eloquent_model(): void
    {
        $productStatistics = new ProductStatistics();
        
        // Sprawdzenie czy klasa dziedziczy po Illuminate Database Eloquent Model
        $this->assertInstanceOf(\Illuminate\Database\Eloquent\Model::class, $productStatistics);
    }

    /**
     * Test sprawdzający czy klasa używa HasFactory trait
     */
    public function test_uses_has_factory_trait(): void
    {
        $reflection = new ReflectionClass(ProductStatistics::class);
        $traits = $reflection->getTraitNames();
        
        $this->assertContains('Illuminate\Database\Eloquent\Factories\HasFactory', $traits);
    }

    /**
     * Test konstruktora z danymi
     */
    public function test_constructor_with_attributes(): void
    {
        $attributes = [
            'employer_id' => 1,
            'product_id' => 1,
            'quantity' => 10,
            'price' => '99.99',
            'valid_from' => '2025-01-01',
            'valid_to' => '2025-12-31',
            'status' => 'active'
        ];

        // Test czy konstruktor przyjmuje tablicę atrybutów
        $productStatistics = new ProductStatistics($attributes);
        $this->assertInstanceOf(ProductStatistics::class, $productStatistics);
    }

    /**
     * Test sprawdzający czy wszystkie wymagane właściwości są zdefiniowane
     */
    public function test_all_required_properties_are_defined(): void
    {
        $productStatistics = new ProductStatistics();
        $reflection = new ReflectionClass($productStatistics);
        
        $expectedProperties = ['table', 'fillable'];
        
        foreach ($expectedProperties as $property) {
            $this->assertTrue(
                $reflection->hasProperty($property),
                "Właściwość {$property} nie została znaleziona w klasie ProductStatistics"
            );
        }
    }

    /**
     * Test sprawdzający dostępność metod relacyjnych
     */
    public function test_relationship_methods_exist(): void
    {
        $productStatistics = new ProductStatistics();
        $reflection = new ReflectionClass($productStatistics);
        
        $expectedMethods = ['employer', 'product'];
        
        foreach ($expectedMethods as $method) {
            $this->assertTrue(
                $reflection->hasMethod($method),
                "Metoda {$method} nie została znaleziona w klasie ProductStatistics"
            );
        }
    }

    /**
     * Test sprawdzający czy metody relacji belongsTo istnieją
     */
    public function test_belongs_to_relationships(): void
    {
        $reflection = new ReflectionClass(ProductStatistics::class);
        
        // Test czy metoda employer istnieje i jest publiczna
        $employerMethod = $reflection->getMethod('employer');
        $this->assertTrue($employerMethod->isPublic());
        
        // Test czy metoda product istnieje i jest publiczna
        $productMethod = $reflection->getMethod('product');
        $this->assertTrue($productMethod->isPublic());
        
        // Sprawdzenie czy metody nie wymagają parametrów
        $this->assertCount(0, $employerMethod->getParameters());
        $this->assertCount(0, $productMethod->getParameters());
    }

    /**
     * Test sprawdzający dostępność metod scope
     */
    public function test_scope_methods_exist(): void
    {
        $productStatistics = new ProductStatistics();
        $reflection = new ReflectionClass($productStatistics);
        
        $expectedScopes = ['scopeActive', 'scopeByEmployer', 'scopeByProduct'];
        
        foreach ($expectedScopes as $scope) {
            $this->assertTrue(
                $reflection->hasMethod($scope),
                "Metoda scope {$scope} nie została znaleziona w klasie ProductStatistics"
            );
        }
    }

    /**
     * Test sprawdzający sygnatury metod scope
     */
    public function test_scope_methods_signatures(): void
    {
        $reflection = new ReflectionClass(ProductStatistics::class);
        
        // Test metody scopeActive
        $activeScope = $reflection->getMethod('scopeActive');
        $this->assertTrue($activeScope->isPublic());
        $this->assertCount(1, $activeScope->getParameters());
        
        // Test metody scopeByEmployer
        $employerScope = $reflection->getMethod('scopeByEmployer');
        $this->assertTrue($employerScope->isPublic());
        $this->assertCount(2, $employerScope->getParameters());
        
        // Test metody scopeByProduct
        $productScope = $reflection->getMethod('scopeByProduct');
        $this->assertTrue($productScope->isPublic());
        $this->assertCount(2, $productScope->getParameters());
    }

    /**
     * Test sprawdzający czy wszystkie pola fillable są stringami lub właściwymi typami
     */
    public function test_fillable_field_types(): void
    {
        $productStatistics = new ProductStatistics();
        $reflection = new ReflectionClass($productStatistics);
        
        $fillableProperty = $reflection->getProperty('fillable');
        $fillableProperty->setAccessible(true);
        $fillable = $fillableProperty->getValue($productStatistics);
        
        // Sprawdzenie czy wszystkie fillable są stringami
        foreach ($fillable as $field) {
            $this->assertIsString($field, "Pole fillable '{$field}' powinno być stringiem");
        }
        
        // Sprawdzenie czy nie ma zduplikowanych pól
        $this->assertEquals(count($fillable), count(array_unique($fillable)), 'Nie powinno być zduplikowanych pól w fillable');
    }

    /**
     * Test sprawdzający czy istnieją wymagane pola biznesowe
     */
    public function test_required_business_fields_are_fillable(): void
    {
        $productStatistics = new ProductStatistics();
        $reflection = new ReflectionClass($productStatistics);
        
        $fillableProperty = $reflection->getProperty('fillable');
        $fillableProperty->setAccessible(true);
        $fillable = $fillableProperty->getValue($productStatistics);
        
        $requiredBusinessFields = ['employer_id', 'product_id', 'quantity', 'price', 'status'];
        
        foreach ($requiredBusinessFields as $field) {
            $this->assertContains($field, $fillable, "Wymagane pole biznesowe '{$field}' nie jest w fillable");
        }
    }

    /**
     * Test sprawdzający czy metoda casts istnieje
     */
    public function test_casts_method_exists(): void
    {
        $reflection = new ReflectionClass(ProductStatistics::class);
        
        $this->assertTrue($reflection->hasMethod('casts'));
        
        $castsMethod = $reflection->getMethod('casts');
        $this->assertTrue($castsMethod->isProtected());
        $this->assertEquals('array', $castsMethod->getReturnType()?->getName());
    }
}