<?php

namespace MbeGroup\Product\Filament\Resources;

use MbeGroup\Product\Models\Product;

use Filament\Resources\Resource;
use Filament\Schemas\Schema;
use Filament\Tables;
use Filament\Tables\Table;
use MbeGroup\Product\Filament\Resources\ProductResource\Pages\ListProducts;
use MbeGroup\Product\Filament\Resources\ProductResource\Pages\CreateProduct;
use MbeGroup\Product\Filament\Resources\ProductResource\Pages\EditProduct;
use Filament\Actions;
use Filament\Forms;
use App\ValueObject\Status;

class ProductResource extends Resource
{
    protected static ?string $model = Product::class;

    public static function getNavigationGroup(): string
    {
        return __('product::filament.navigation.group.label');
    }

    public static function getNavigationLabel(): string
    {
        return __('product::filament.resources.product.navigation_label');
    }

    public static function getModelLabel(): string
    {
        return __('product::filament.resources.product.model_label');
    }

    public static function getPluralModelLabel(): string
    {
        return __('product::filament.resources.product.plural_model_label');
    }

    public static function form(Schema $schema): Schema
    {
        return $schema
            ->schema([
                Forms\Components\TextInput::make('name')
                    ->label(__('product::filament.resources.product.form.fields.name'))
                    ->required()
                    ->maxLength(255),
                Forms\Components\TextInput::make('price_net')
                    ->label(__('product::filament.resources.product.form.fields.price_net'))
                    ->required()
                    ->numeric(),
                Forms\Components\Select::make('category')
                    ->label(__('product::filament.resources.product.form.fields.category'))
                    ->required()
                    ->options(Product::CATEGORIES),
                Forms\Components\Select::make('unit')
                    ->label(__('product::filament.resources.product.form.fields.unit'))
                    ->required()
                    ->options(Product::UNITS),
                Forms\Components\TextInput::make('quantity')
                    ->label(__('product::filament.resources.product.form.fields.quantity'))
                    ->required()
                    ->numeric(),
                Forms\Components\Select::make('status')
                        ->label(__('product::filament.resources.product.form.fields.status'))
                        ->options(Status::getLabels())
                        ->required()
                        ->default(Status::INACTIVE)
                        ->afterStateHydrated(function (Forms\Components\Select $component, $state): void {
                            if ($state instanceof Status) {
                                $component->state($state->value);
                            }
                        })
                        ->dehydrateStateUsing(function ($state): int {
                            return $state instanceof Status ? $state->value: (int) $state;
                        })
                        ->rule('integer')
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('id')
                    ->label('ID')
                    ->sortable(),
                Tables\Columns\TextColumn::make('name')
                    ->label(__('product::filament.resources.product.table.columns.name'))
                    ->searchable(),
                Tables\Columns\TextColumn::make('price_net')
                    ->label(__('product::filament.resources.product.table.columns.price_net'))
                    ->money('PLN', true)
                    ->sortable(),
                Tables\Columns\IconColumn::make('status')
                    ->label(__('product::filament.resources.product.table.columns.status'))
                    ->state(fn ($record) => Status::int($record->status))
                    ->icon(fn ($state) => Status::icon($state))
                    ->color(fn ($state) => Status::color($state))
                    ->tooltip(fn ($state) => Status::label($state))
                    ->sortable(),
                Tables\Columns\TextColumn::make('created_at')
                ->label(__('product::filament.resources.product.table.columns.created_at'))
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
                Tables\Columns\TextColumn::make('updated_at')
                ->label(__('product::filament.resources.product.table.columns.updated_at'))
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                //
            ])
            ->recordActions([
                Actions\EditAction::make(),
                Actions\DeleteAction::make(),
            ])
            ->toolbarActions([
                Actions\BulkActionGroup::make([
                    Actions\DeleteBulkAction::make(),
                ]),
            ])
            ->description('Version: ' . \Composer\InstalledVersions::getVersion('mbegroup/product'));
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => ListProducts::route('/'),
            'create' => CreateProduct::route('/create'),
            'edit' => EditProduct::route('/{record}/edit'),
        ];
    }
}
