<?php

namespace Tests\Unit;

use MbeGroup\Product\Filament\ProductPlugin;
use PHPUnit\Framework\TestCase;
use ReflectionClass;

class ProductPluginTest extends TestCase
{
    /**
     * Test tworzenia nowej instancji ProductPlugin
     */
    public function test_can_create_product_plugin_instance(): void
    {
        $plugin = new ProductPlugin();
        
        $this->assertInstanceOf(ProductPlugin::class, $plugin);
    }

    /**
     * Test sprawdzający czy klasa implementuje Plugin interface
     */
    public function test_implements_plugin_interface(): void
    {
        $plugin = new ProductPlugin();
        
        $this->assertInstanceOf(\Filament\Contracts\Plugin::class, $plugin);
    }

    /**
     * Test sprawdzający namespace klasy
     */
    public function test_class_namespace(): void
    {
        $reflection = new ReflectionClass(ProductPlugin::class);
        
        $this->assertEquals('MbeGroup\Product\Filament', $reflection->getNamespaceName());
        $this->assertEquals('ProductPlugin', $reflection->getShortName());
    }

    /**
     * Test sprawdzający czy metoda getId zwraca poprawny identyfikator
     */
    public function test_get_id_method(): void
    {
        $plugin = new ProductPlugin();
        
        $this->assertEquals('product', $plugin->getId());
        $this->assertIsString($plugin->getId());
    }

    /**
     * Test sprawdzający czy wszystkie wymagane metody są zdefiniowane
     */
    public function test_required_methods_are_defined(): void
    {
        $reflection = new ReflectionClass(ProductPlugin::class);
        
        $requiredMethods = ['getId', 'register', 'boot'];
        
        foreach ($requiredMethods as $method) {
            $this->assertTrue(
                $reflection->hasMethod($method),
                "Metoda {$method} nie została znaleziona w klasie ProductPlugin"
            );
        }
    }

    /**
     * Test sprawdzający sygnatury metod interface Plugin
     */
    public function test_plugin_interface_methods_signatures(): void
    {
        $reflection = new ReflectionClass(ProductPlugin::class);
        
        // Test metody getId
        $getIdMethod = $reflection->getMethod('getId');
        $this->assertTrue($getIdMethod->isPublic());
        $this->assertEquals('string', $getIdMethod->getReturnType()?->getName());
        
        // Test metody register
        $registerMethod = $reflection->getMethod('register');
        $this->assertTrue($registerMethod->isPublic());
        $this->assertEquals('void', $registerMethod->getReturnType()?->getName());
        $this->assertCount(1, $registerMethod->getParameters());
        
        // Test metody boot
        $bootMethod = $reflection->getMethod('boot');
        $this->assertTrue($bootMethod->isPublic());
        $this->assertEquals('void', $bootMethod->getReturnType()?->getName());
        $this->assertCount(1, $bootMethod->getParameters());
    }

    /**
     * Test sprawdzający czy register metoda przyjmuje Panel jako parametr
     */
    public function test_register_method_parameter_type(): void
    {
        $reflection = new ReflectionClass(ProductPlugin::class);
        $registerMethod = $reflection->getMethod('register');
        $parameters = $registerMethod->getParameters();
        
        $this->assertEquals('panel', $parameters[0]->getName());
        $this->assertEquals('Filament\Panel', $parameters[0]->getType()?->getName());
    }

    /**
     * Test sprawdzający czy boot metoda przyjmuje Panel jako parametr
     */
    public function test_boot_method_parameter_type(): void
    {
        $reflection = new ReflectionClass(ProductPlugin::class);
        $bootMethod = $reflection->getMethod('boot');
        $parameters = $bootMethod->getParameters();
        
        $this->assertEquals('panel', $parameters[0]->getName());
        $this->assertEquals('Filament\Panel', $parameters[0]->getType()?->getName());
    }

    /**
     * Test sprawdzający czy ID pluginu nie jest pusty
     */
    public function test_plugin_id_is_not_empty(): void
    {
        $plugin = new ProductPlugin();
        $id = $plugin->getId();
        
        $this->assertNotEmpty($id);
        $this->assertIsString($id);
        $this->assertGreaterThan(0, strlen($id));
    }
}