<?php

namespace Tests\Unit;

use MbeGroup\Product\Models\Product;
use PHPUnit\Framework\TestCase;
use ReflectionClass;

class ProductTest extends TestCase
{
    /**
     * Test tworzenia nowej instancji Product
     */
    public function test_can_create_product_instance(): void
    {
        $product = new Product();

        $this->assertInstanceOf(Product::class, $product);
    }

    /**
     * Test sprawdzający czy klasa ma odpowiednie właściwości
     */
    public function test_product_class_properties(): void
    {
        $reflection = new ReflectionClass(Product::class);

        // Sprawdzenie czy klasa dziedziczy po Illuminate\Database\Eloquent\Model
        $this->assertTrue($reflection->hasProperty('table'));
        $this->assertTrue($reflection->hasProperty('fillable'));
    }

    /**
     * Test sprawdzający wypełnialne pola
     */
    public function test_fillable_attributes(): void
    {
        $product = new Product();
        $reflection = new ReflectionClass($product);

        $fillableProperty = $reflection->getProperty('fillable');
        $fillableProperty->setAccessible(true);
        $fillable = $fillableProperty->getValue($product);
        $expectedFillable = [
            'name',
            'price_net',
            'category',
            'quantity',
            'status',
            'validity_period_months',
            'product_life_months',
        ];
        $this->assertEquals($expectedFillable, $fillable);
    }

    /**
     * Test sprawdzający nazwę tabeli
     */
    public function test_table_name(): void
    {
        $product = new Product();
        $reflection = new ReflectionClass($product);

        $tableProperty = $reflection->getProperty('table');
        $tableProperty->setAccessible(true);
        $table = $tableProperty->getValue($product);

        $this->assertEquals('products', $table);
    }

    /**
     * Test sprawdzający namespace klasy
     */
    public function test_class_namespace(): void
    {
        $product = new Product();
        $reflection = new ReflectionClass($product);

        $this->assertEquals('MbeGroup\Product\Models', $reflection->getNamespaceName());
        $this->assertEquals('Product', $reflection->getShortName());
    }

    /**
     * Test czy klasa implementuje odpowiednie metody (sprawdzenie czy dziedziczy po Model)
     */
    public function test_extends_eloquent_model(): void
    {
        $product = new Product();

        // Sprawdzenie czy klasa dziedziczy po Illuminate Database Eloquent Model
        $this->assertInstanceOf(\Illuminate\Database\Eloquent\Model::class, $product);
    }

    /**
     * Test sprawdzający czy klasa używa HasFactory trait
     */
    public function test_uses_has_factory_trait(): void
    {
        $reflection = new ReflectionClass(Product::class);
        $traits = $reflection->getTraitNames();

        $this->assertContains('Illuminate\Database\Eloquent\Factories\HasFactory', $traits);
    }

    /**
     * Test konstruktora z danymi
     */
    public function test_constructor_with_attributes(): void
    {
        $attributes = [
            'name' => 'Test Product',
            'price_net' => '99.99',
            'category' => 'product-job-offers',
            'unit' => 'sztuka',
            'quantity' => 10,
            'status' => 'active'
        ];

        // Test czy konstruktor przyjmuje tablicę atrybutów
        $product = new Product($attributes);
        $this->assertInstanceOf(Product::class, $product);
    }

    /**
     * Test sprawdzający czy wszystkie wymagane właściwości są zdefiniowane
     */
    public function test_all_required_properties_are_defined(): void
    {
        $product = new Product();
        $reflection = new ReflectionClass($product);

        $expectedProperties = ['table', 'fillable'];

        foreach ($expectedProperties as $property) {
            $this->assertTrue(
                $reflection->hasProperty($property),
                "Właściwość {$property} nie została znaleziona w klasie Product"
            );
        }
    }

    /**
     * Test sprawdzający dostępność metod relacyjnych
     */
    public function test_relationship_methods_exist(): void
    {
        $product = new Product();
        $reflection = new ReflectionClass($product);

        $expectedMethods = ['productOrders', 'employers'];

        foreach ($expectedMethods as $method) {
            $this->assertTrue(
                $reflection->hasMethod($method),
                "Metoda {$method} nie została znaleziona w klasie Product"
            );
        }
    }

    /**
     * Test sprawdzający czy metody relacji hasMany istnieją
     */
    public function test_has_many_relationships(): void
    {
        $reflection = new ReflectionClass(Product::class);

        // Test czy metoda productOrders istnieje i jest publiczna
        $productOrdersMethod = $reflection->getMethod('productOrders');
        $this->assertTrue($productOrdersMethod->isPublic());

        // Sprawdzenie czy metoda nie wymaga parametrów
        $this->assertCount(0, $productOrdersMethod->getParameters());
    }

    /**
     * Test sprawdzający czy metody relacji hasManyThrough istnieją
     */
    public function test_has_many_through_relationships(): void
    {
        $reflection = new ReflectionClass(Product::class);

        // Test czy metoda employers istnieje i jest publiczna
        $employersMethod = $reflection->getMethod('employers');
        $this->assertTrue($employersMethod->isPublic());

        // Sprawdzenie czy metoda nie wymaga parametrów
        $this->assertCount(0, $employersMethod->getParameters());
    }

    /**
     * Test sprawdzający czy wszystkie pola fillable są stringami lub właściwymi typami
     */
    public function test_fillable_field_types(): void
    {
        $product = new Product();
        $reflection = new ReflectionClass($product);

        $fillableProperty = $reflection->getProperty('fillable');
        $fillableProperty->setAccessible(true);
        $fillable = $fillableProperty->getValue($product);

        // Sprawdzenie czy wszystkie fillable są stringami
        foreach ($fillable as $field) {
            $this->assertIsString($field, "Pole fillable '{$field}' powinno być stringiem");
        }

        // Sprawdzenie czy nie ma zduplikowanych pól
        $this->assertEquals(count($fillable), count(array_unique($fillable)), 'Nie powinno być zduplikowanych pól w fillable');
    }

    /**
     * Test sprawdzający czy istnieją wymagane pola biznesowe
     */
    public function test_required_business_fields_are_fillable(): void
    {
        $product = new Product();
        $reflection = new ReflectionClass($product);

        $fillableProperty = $reflection->getProperty('fillable');
        $fillableProperty->setAccessible(true);
        $fillable = $fillableProperty->getValue($product);

        $requiredBusinessFields = ['name', 'price_net', 'status'];

        foreach ($requiredBusinessFields as $field) {
            $this->assertContains($field, $fillable, "Wymagane pole biznesowe '{$field}' nie jest w fillable");
        }
    }

    /**
     * Test sprawdzający konstante kategorii
     */
    public function test_categories_constant(): void
    {
        $reflection = new ReflectionClass(Product::class);

        $this->assertTrue($reflection->hasConstant('CATEGORIES'));

        $categories = $reflection->getConstant('CATEGORIES');
        $this->assertIsArray($categories);
        $this->assertArrayHasKey('product-job-offers', $categories);
        $this->assertArrayHasKey('product-image', $categories);
    }

    /**
     * Test sprawdzający czy metoda casts istnieje
     */
    public function test_casts_method_exists(): void
    {
        $reflection = new ReflectionClass(Product::class);

        $this->assertTrue($reflection->hasMethod('casts'));

        $castsMethod = $reflection->getMethod('casts');
        $this->assertTrue($castsMethod->isProtected());
        $this->assertEquals('array', $castsMethod->getReturnType()?->getName());
    }
}
